-- Ablage von Dokumenten als Bytea um diese zu einem späteren Zeitpunkt im DMS abzulegen
CREATE TABLE picndoku_delayed_upload_pg (
  du_id       SERIAL PRIMARY KEY,
  du_bytea    bytea,              -- Dokument als Bytea
  du_metadata jsonb DEFAULT NULL  -- JSON mit Metadaten z.B. Dateiextension
 );

-- interne Funktion für tsystem.picndoku__delayed_upload__create und tsystem.picndoku__delayed_upload__build
CREATE OR REPLACE FUNCTION tsystem.picndoku__delayed_upload__build_meta(
    IN _file_name   varchar,  -- z.B. 'test.pdf'
    IN _file_source varchar DEFAULT 'file',
    IN _du_id       integer DEFAULT NULL
  )
  RETURNS jsonb
  AS $$
  BEGIN
    RETURN jsonb_build_object
      (
        '_info',              'db-blob pending to be saved to dms',
        'filename_extension', substring( _file_name FROM E'\\.([^.]+)$' ),  -- alles nach dem letzten Punkt, z.B. 'pdf' in 'test.pdf'
        'filename',           _file_name,
        'table_name',         'picndoku_delayed_upload_pg',
        'column_file',        'du_bytea',
        'column_pkey',        'du_id',
        'pkey',               _du_id,
        'source',             _file_source
      );
  END $$ LANGUAGE plpgsql STABLE;
--

--
CREATE OR REPLACE FUNCTION tsystem.picndoku__md5_to_str(hexhash varchar) RETURNS varchar AS $$
  BEGIN
    -- Die ursprünglich verwendete WinAPI MD5Init/MD5Update/MD5Final enthält keine ToString-Funktion.
    -- MD5 und IID/GUID enthalten sind beide 16 Byte groß und somit bot Delphi hierfür eine einfach zu verwendende Funktion, über einen winzigen Cast.
    --
    -- Logisch in etwa so (Auskommentiertes), aber ich hatte es nicht geschafft die Endianness der ersten 3 Integer umzudrehen.
    -- Außerdem kennt Format im Postgres keine Integer/Hex: upper(format('{%8x-%4x-%4x-%4s-%12s}', ...))
    --
    --  7fc56270e7a70fa81a5935b72eacbe29      <- INPUT = md5('A'::BYTEA)
    --  7fc56270 e7a7 0fa8 1a59 35b72eacbe29
    -- {7062C57F-A7E7-A80F-1A59-35B72EACBE29} -> OUTPUT = GUID (16 Bytes)
    -- {Int32-Int16-Int16-2xByte-6xByte}
    /*
    RETURN
      upper('{'
        || lpad(to_hex(('x' || substring(hexhash,  1, 8))::BIT(32)::INT), 8, '0') || '-'
        || lpad(to_hex(('x' || substring(hexhash,  9, 4))::BIT(16)::INT), 4, '0') || '-'
        || lpad(to_hex(('x' || substring(hexhash, 13, 4))::BIT(16)::INT), 4, '0') || '-'
        || substring(hexhash, 17,  4) || '-'
        || substring(hexhash, 21, 12) || '}';
      );
    */
    RETURN
      upper('{'
        || substring(hexhash, 7,  2) || substring(hexhash,  5, 2) || substring(hexhash, 3, 2) || substring(hexhash, 1, 2) || '-'
        || substring(hexhash, 11, 2) || substring(hexhash,  9, 2) || '-'
        || substring(hexhash, 15, 2) || substring(hexhash, 13, 2) || '-'
        || substring(hexhash, 17, 4) || '-'
        || substring(hexhash, 21, 12) || '}'
      );
  END $$ LANGUAGE plpgsql STABLE;
--

-- Datei über Datenbank ins DMS (erstmal als BLOB und anschließend durch APPS ins Dateisystem)
-- z.B. für XRechnung
CREATE OR REPLACE FUNCTION tsystem.picndoku__delayed_upload__create(
    IN _file_bytea     bytea,   -- z.B. pg_read_binary_file('C:\temp\test.pdf')
    IN _file_name      varchar, -- z.B. 'test.pdf'

    IN _tablename      varchar,
    IN _dbrid          varchar,

    IN _dokident       varchar DEFAULT null,
    IN _doktype        varchar DEFAULT null,

    IN _file_source    varchar DEFAULT 'file',
    IN _file_name_user varchar DEFAULT null,  -- nur Dateiname, ohne Pfad
    IN _file_name_orig varchar DEFAULT null,  -- ...

    OUT _du_id         integer,
    OUT _pd_id         integer
  )
  RETURNS record
  AS $$
  DECLARE
    filehash    varchar;
    metadata    jsonb;
    revisionid  integer;
    remotefile  varchar;
  BEGIN
    _file_name_orig := coalesce(_file_name_orig, _file_name);
    IF (nullif(_file_name, '') IS NULL) OR (_file_name ~ E'\\\\|\\/') OR (_file_name_orig ~ E'\\\\|\\/') THEN
      RAISE EXCEPTION '_file_name und _file_name_orig dürfen nur Dateinamen sein, ohne Pfad';
    END IF;

    --Metadaten für File erstellen
    filehash := tsystem.picndoku__md5_to_str( md5(_file_bytea) );
    metadata := tsystem.picndoku__delayed_upload__build_meta(_file_name, _file_source, NULL);

    -- Dokument als Bytea in Datenbank speichern
    INSERT INTO picndoku_delayed_upload_pg
      ( du_bytea   , du_metadata )
    VALUES
      ( _file_bytea, metadata    )
    RETURNING
      du_id
    INTO
      _du_id
    ;

    metadata        := jsonb_set( metadata, '{pkey}', to_jsonb(_du_id) );  -- die ID war oben noch nicht bekannt

    _pd_id          := nextval('picndoku_pd_id_seq'::regclass);  -- nicht über DEFAULT, denn wird bereits für remotefile/pd_dmsremotefile benötigt
    remotefile      := _pd_id || substring( _file_name FROM E'(\\.[^.]+)$' );  -- "ID.Dateiendung" (siehe TDokument.DMSGenerateFileName)

    -- Erstelle picndoku-Eintrag
    INSERT INTO picndoku
      ( pd_id , pd_tablename, pd_dbrid, pd_source   , pd_dokumentfile, pd_path   , pd_path_user   , pd_dokident, pd_doktype, pd_dmshash, pd_dms_delayed_upload, pd_dms_delayed_info, pd_dmsremotefile, pd_dmsremotefiledoktype )
    VALUES
      ( _pd_id, _tablename  , _dbrid  , _file_source, _file_name_orig, _file_name, _file_name_user, _dokident  , _doktype  , filehash  , True                 , metadata           , remotefile      , _doktype                )
    --RETURNING
    --  pd_id
    --INTO
    --  _pd_id    <- siehe _pd_id:=nextval(...
    ;

    -- Revision/Sperre erstellen und verknüpfen
    -- eigentlich jetzt absolut sinnlos, aber soll so bleiben (DS)          jetzt im BLOB, aber früher als verlinkte&ausgecheckte Datei (darum war mal mit Revision+Sperre)
    INSERT INTO picndoku_revision
      ( pdr_revision_id, pdr_blocked_pd_id, pdr_blocked_date, pdr_blocked_by, pdr_checkout_path, pdr_checkout_ask_user )
    VALUES
      ( _pd_id         , _pd_id           , now()           , current_user  , '<DELAYED>'      , False                 )
    ;
    UPDATE picndoku
       SET pd_revision_id = _pd_id
     WHERE pd_id = _pd_id
    ;

    RETURN;

  END $$ LANGUAGE plpgsql VOLATILE;
--

-- Datei über Datenbank ins DMS (erstmal als BLOB und anschließend durch APPS ins Dateisystem)
-- für TDokument.DMSSetServerFile, falls APPS nicht erreichbar oder Schreibfehler
CREATE OR REPLACE FUNCTION tsystem.picndoku__delayed_upload__build(
    IN _pd_id          integer,
    IN _pd_revision_id integer,
    IN _file_bytea     bytea,    -- z.B. pg_read_binary_file('C:\temp\test.pdf')
    IN _file_name      varchar,  -- z.B. 'test.pdf'
    IN _file_source    varchar DEFAULT 'file',

    OUT _old_du_id     integer,  -- nach erfolgreichem Speichern (clientseitig) den alten BLOB löschen, falls vorhanden
    OUT _du_id         integer,  -- dieses Feld clientseitig ins TDataSet übernehmen (INSERT/UPDATE)
    OUT _delayed_info  text      -- dieses Feld clientseitig ins TDataSet übernehmen (INSERT/UPDATE)
  )
  RETURNS record
  AS $$
  DECLARE
    metadata jsonb;
  BEGIN

    -- aktuelle eine Sperre entfernen (gesperrte/ausgecheckte Datei oder altes DMSDelayedUpload)
    IF (SELECT True
        FROM   picndoku_revision
        WHERE  pdr_revision_id = _pd_revision_id
          AND  pdr_blocked_by IS NOT NULL
       )
    THEN
      PERFORM picndoku__lock_or_checkout__remove(NULL, _pd_id, 0, True);
    END IF;

    -- Falls bereits pd_dms_delayed_upload vorhanden, dann die du_id des alten BLOBs zurückgeben.
    -- Wird vom Aufrufer gelöscht, NACHDEM das Speichern erfolgreich abgeschlossen wurde.
    _old_du_id := (
      SELECT CASE
               WHEN pd_dms_delayed_info IS NULL
               THEN NULL
               ELSE pd_dms_delayed_info::json ->> 'pkey'
             END
        FROM picndoku
       WHERE pd_id = _pd_id
    );

    -- Metadaten für File erstellen
    metadata := tsystem.picndoku__delayed_upload__build_meta(_file_name, _file_source, NULL);

    -- Dokument als Bytea in Datenbank speichern
    INSERT INTO picndoku_delayed_upload_pg
      ( du_bytea   , du_metadata )
    VALUES
      ( _file_bytea, metadata    )
    RETURNING
      du_id
    INTO
      _du_id
    ;

    -- Daten für Delayed Upload Prozess vorbereiten
    metadata      := jsonb_set( metadata, '{pkey}', to_jsonb(_du_id) );  -- die ID war oben noch nicht bekannt
    _delayed_info := metadata;

    RETURN;

  END $$ LANGUAGE plpgsql VOLATILE;
--

-- Daten liefern, womit der APPS das DMSDelayedUpload auflöst (Datei auf Festplatte schreibt), sowie der Client (TDokument.DMSGetFile) die Datei aus dem BLOB holen kann.
-- Ebenso, um den BLOB zu laden (TDokument.DMSGetFile/DMSGetServerFile), wenn bereits auf die Datei zugegriffen wird, bevor der Apps die Datei speichern konnte.
CREATE OR REPLACE FUNCTION tsystem.picndoku__delayed_upload__get(
    IN  _picndoku picndoku,
    IN  _raise_exception boolean DEFAULT true,

    OUT _du_table varchar,  -- JSON.table_name    Wird genutzt, im nach erfolgreichem Speichern den BLOB zu löschen.
    OUT _du_pkey  varchar,  -- JSON.column_pkey   ...
    OUT _du_id    integer,  -- JSON.pkey          ...
    OUT _du_bytea bytea     -- Dateiinhalt
  )
  RETURNS record
  AS $$
  DECLARE
    sql_query   varchar;
    datafield   varchar;
    du_metadata jsonb;
  BEGIN
    IF _raise_exception AND _picndoku IS NULL THEN
      RAISE EXCEPTION 'Kein Dokument angegeben';
    END IF;

    -- Tabelle und Primärschlüssel ermitteln

    _du_table  := _picndoku.pd_dms_delayed_info::json->>'table_name';
    _du_pkey   := _picndoku.pd_dms_delayed_info::json->>'column_pkey';
    _du_id     := (_picndoku.pd_dms_delayed_info::json->>'pkey')::integer;
    datafield  := _picndoku.pd_dms_delayed_info::json->>'column_file';

    IF (_du_table IS NULL OR _du_pkey IS NULL OR _du_id IS NULL OR datafield IS NULL) THEN
      IF _raise_exception THEN
        RAISE EXCEPTION 'Dokument nicht gefunden oder ungültig: pd_id %', _pd_id;
      END IF;
    ELSE
      -- Datei und Metadaten abrufen (dynamische SQL-Anweisung vorbereiten und ausführen)
      sql_query := format( 'SELECT %I, du_metadata FROM %I WHERE %I = %L', datafield, _du_table, _du_pkey, _du_id );
      EXECUTE sql_query INTO _du_bytea, du_metadata;

      IF _raise_exception AND _du_bytea IS NULL THEN
        RAISE EXCEPTION 'Fehler beim Laden des Dateiinhalts: Tabelle %, Schlüssel %', _du_table, _du_id;
      END IF;
    END IF;

    RETURN;
  END $$ LANGUAGE plpgsql VOLATILE;
--

--
CREATE OR REPLACE FUNCTION tsystem.picndoku__delayed_upload__get(
    IN  _pd_id    integer,

    OUT _du_table varchar,  -- JSON.table_name    Wird genutzt, im nach erfolgreichem Speichern den BLOB zu löschen.
    OUT _du_pkey  varchar,  -- JSON.column_pkey   ...
    OUT _du_id    integer,  -- JSON.pkey          ...
    OUT _du_bytea bytea     -- Dateiinhalt
  )
  RETURNS record
  AS $$
  BEGIN

    -- Tabelle und Primärschlüssel ermitteln
    SELECT (tsystem.picndoku__delayed_upload__get(picndoku)).*
      INTO _du_table, _du_pkey, _du_id, _du_bytea
      FROM picndoku
     WHERE pd_id = _pd_id;

    RETURN;

  END $$ LANGUAGE plpgsql VOLATILE;
--

-- DMSDelayedUpload "jetzt" in DMS-Verzeichnichs kopieren : für manuellen Aufruf aus Client/WebApp (falls die Datei physisch sofort verfügbar sein muß)
-- z.B. gerade eben via tsystem.picndoku__delayed_upload__create erstellt.
-- TODO : neue REST-Funktion in APPS, welche nur die pd_id bekommt und alles innerhalb des APPS via TDokument erledigt, ohne den BLOB zu übertragen (entsprechend der TProdatSrvXEService.DMSDelayedUpload)
CREATE OR REPLACE FUNCTION tsystem.picndoku__delayed_upload__export(
    _pd_id       integer,
    DMS_API_HOST varchar DEFAULT coalesce(TSystem.Settings__Get('DMS_API_HOST'), 'localhost'),
    DMS_API_PORT varchar DEFAULT coalesce(TSystem.Settings__Get('DMS_API_PORT'), '212'),
    AUTH_HEADER  varchar DEFAULT 'Basic SW52b2ljZVVwbG9hZDo3Q0IyNzcyQS03Njg0LTQxNjYtQkM5MC1EQjAwMTYwNUZFQzU='  -- TODO: in WindowsCredentials verschieben
  )
  RETURNS json
  AS $$
  DECLARE
    database_connection varchar;
    apps_connection     varchar;
    uri                 varchar;
    table_name          varchar;
    pkey                varchar;
    column_file         varchar;
    column_pkey         varchar;
    file_base64         varchar;
    sql_query           varchar;
    content_json        json;
    doc_metadata        jsonb;
    result              record;
  BEGIN

    -- Aktuelle Datenbankverbindung abrufen
    database_connection := ( SELECT concat( current_database(), '@localhost:', inet_server_port() ) );

    -- Verbindung zur APPS-Datenbank prüfen
    uri := CONCAT( DMS_API_HOST, ':', DMS_API_PORT, '/DataSnap/REST/TDSFunctionMethods/DBServer' );
    apps_connection := tsystem.apps_connection__get( uri, AUTH_HEADER );

    IF apps_connection <> database_connection THEN
      RAISE EXCEPTION 'APPS ist zur falschen Datenbank verbunden. Aktuelle Datenbank: % | APPS: %', database_connection, apps_connection;
    END IF;

    -- Tabelle und Primärschlüssel ermitteln
    SELECT pd_dms_delayed_info::json ->> 'table_name'  AS table_name,
           pd_dms_delayed_info::json ->> 'pkey'        AS pkey,
           pd_dms_delayed_info::json ->> 'column_file' AS column_file,
           pd_dms_delayed_info::json ->> 'column_pkey' AS column_pkey
      INTO table_name, pkey, column_file, column_pkey
      FROM picndoku
     WHERE pd_id = _pd_id;

    IF table_name IS NULL OR pkey IS NULL THEN
      RAISE EXCEPTION 'Dokument nicht gefunden oder ungültige pd_id: %', _pd_id;
    END IF;

    -- Datei Base64 enkodieren und Metadaten abrufen
    -- dynamische SQL-Anweisung vorbereiten ...
    sql_query := format( 'SELECT encode(%I, ''base64''), du_metadata FROM %I WHERE %I = %L', column_file, table_name, column_pkey, pkey );
    -- ... und ausführen
    EXECUTE sql_query INTO file_base64, doc_metadata;

    IF file_base64 IS NULL THEN
        RAISE EXCEPTION 'Fehler beim Enkodieren der Datei. Tabelle: %, Schlüssel: %', table_name, pkey;
    END IF;

    -- Datei ins DMS hochladen
    uri := concat( DMS_API_HOST, ':', DMS_API_PORT, '/DataSnap/REST/TDSFileMethods/%22WriteFileByID%22/', _pd_id );
    content_json := jsonb_build_object( '_parameters', json_build_array( file_base64 ) );

    SELECT *
      INTO result
      FROM tsystem.http_request((
            'POST',
            uri,
            ARRAY[http_header('Authorization', AUTH_HEADER)],
            'text/plain;charset=UTF-8',
            content_json
           )::http_request);
    -- Wenn erfolgreich übertragen, wurden im APPS pd_dms_delayed_upload und pdr_blocked_pd_id/pdr_blocked_date/pdr_blocked_by/pdr_checkout_path geNULLt.

    -- Ergebnis zurückgeben
    RETURN row_to_json(result);

  END $$ LANGUAGE plpgsql VOLATILE;
--

